<?php
require __DIR__ . '/../db.php';
$u = current_user($db);
$isAdmin = $u && $u['role']==='admin';
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>Admin – FeatureBoard</title>
<style>
body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu;margin:0;background:#f6f7fb;color:#111827}
.wrap{max-width:1100px;margin:24px auto;padding:0 16px}
.card{background:#fff;border:1px solid #e5e7eb;border-radius:12px;box-shadow:0 2px 6px rgba(0,0,0,.04);padding:16px;margin-bottom:16px}
.row{display:flex;gap:12px;align-items:center;flex-wrap:wrap}
.btn{border:1px solid #d1d5db;background:#fff;border-radius:10px;padding:6px 12px;cursor:pointer;font-size:14px}
.btn:hover{background:#f3f4f6}
input,select,textarea{border:1px solid #d1d5db;border-radius:10px;padding:6px 8px;font-size:14px}
.muted{color:#6b7280}
.tabs{display:flex;gap:8px;margin-bottom:12px}
.tabbtn{padding:6px 10px;border-radius:999px;border:1px solid #d1d5db;background:#f9fafb;cursor:pointer;font-size:14px}
.tabbtn.active{background:#111827;color:#f9fafb;border-color:#111827}
.tab{display:none}
.tab.active{display:block}
table{border-collapse:collapse;width:100%;font-size:13px}
th,td{border-bottom:1px solid #e5e7eb;padding:6px 4px;text-align:left}
th{font-weight:600}
.badge{font-size:11px;border-radius:999px;border:1px solid #e5e7eb;padding:2px 6px}
.status-badge-1{background:#eff6ff;border-color:#bfdbfe}
.status-badge-2{background:#ecfeff;border-color:#bae6fd}
.status-badge-3{background:#fef9c3;border-color:#facc15}
.status-badge-4{background:#dcfce7;border-color:#86efac}
.status-badge-5{background:#fee2e2;border-color:#fecaca}
.mb8{margin-bottom:8px}
.edit-card{background:#f9fafb;border-radius:10px;border:1px solid #e5e7eb;padding:10px;font-size:13px}
.edit-row-grid{display:grid;grid-template-columns:minmax(0,2fr) minmax(0,1fr);gap:12px}
@media (max-width:700px){
  .edit-row-grid{grid-template-columns:1fr;}
}
</style>
</head>
<body>
<div class="wrap">
  <h1>Admin</h1>

  <div id="login-card" class="card" style="<?php echo $isAdmin ? 'display:none;' : '' ?>">
    <h2>Admin login</h2>
    <p class="muted">Sign in as <strong>FeatureAdmin</strong> with the default password on first run, then change or create your own admin accounts.</p>
    <div class="row mb8">
      <input id="adm-ident" placeholder="FeatureAdmin or admin email" />
      <input id="adm-pass" placeholder="Password" type="password" />
      <button class="btn" id="btn-adm-login">Login</button>
    </div>
    <div id="adm-msg" class="muted"></div>
  </div>

  <div id="admin-ui" style="<?php echo $isAdmin ? '' : 'display:none;' ?>">

    <div class="card">
      <div class="row mb8" style="justify-content:space-between;">
        <div class="muted">Logged in as <strong><?php echo $isAdmin ? htmlspecialchars($u['email']) : '' ?></strong></div>
        <button class="btn" id="btn-logout">Log out</button>
      </div>
      <div class="tabs">
        <button class="tabbtn active" data-tab="settings">Settings</button>
        <button class="tabbtn" data-tab="features">Features</button>
        <button class="tabbtn" data-tab="users">Users</button>
        <button class="tabbtn" data-tab="roadmap">Roadmap</button>
      </div>
    </div>

    <div class="card tab active" id="tab-settings">
      <h2>Settings</h2>
      <h3>Allowed email domain</h3>
      <div class="row mb8">
        <input id="set-domain" style="max-width:260px" />
        <button class="btn" id="btn-save-domain">Save</button>
        <span class="muted" id="set-domain-msg"></span>
      </div>

      <h3>Categories</h3>
      <div class="row mb8">
        <input id="cat-name" placeholder="Category name" />
        <input id="cat-sort" placeholder="Sort" style="max-width:80px" />
        <button class="btn" id="btn-add-cat">Add</button>
      </div>
      <table id="cat-table">
        <thead><tr><th>Name</th><th>Sort</th><th></th></tr></thead>
        <tbody></tbody>
      </table>
    </div>

    <div class="card tab" id="tab-features">
      <h2>Features</h2>
      <div class="row mb8">
        <select id="filt-status">
          <option value="">All statuses</option>
          <option value="1">Backlog</option>
          <option value="2">Planned</option>
          <option value="3">In Progress</option>
          <option value="4">Released</option>
          <option value="5">Rejected</option>
        </select>
        <select id="filt-cat"></select>
        <input id="filt-q" placeholder="Search title" style="min-width:200px" />
        <button class="btn" id="btn-filt">Apply</button>
      </div>
      <table id="feat-table">
        <thead>
          <tr>
            <th>ID</th><th>Title</th><th>Status</th><th>Category</th><th>Votes</th><th>Author</th><th>Dup Of</th><th>Actions</th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
    </div>

    <div class="card tab" id="tab-users">
      <h2>Users</h2>
      <h3>Create admin</h3>
      <div class="row mb8">
        <input id="u-email" placeholder="email@example.com" />
        <input id="u-name" placeholder="Name (optional)" />
        <input id="u-pass" placeholder="Password" type="password" />
        <button class="btn" id="btn-create-admin">Create admin</button>
      </div>
      <h3>All users</h3>
      <table id="user-table">
        <thead>
          <tr>
            <th>Email</th><th>Admin</th><th>Last seen</th>
            <th>B</th><th>P</th><th>IP</th><th>R</th><th>Rej</th><th></th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
      <div id="user-detail" style="margin-top:16px;"></div>
    </div>

    <div class="card tab" id="tab-roadmap">
      <h2>Roadmap</h2>
      <p class="muted">Opens the public roadmap view with columns Backlog, Planned, In Progress, Released.</p>
      <a class="btn" href="roadmap.php" target="_blank">Open Roadmap</a>
    </div>

  </div>
</div>

<script>
let CSRF = null;
async function getCsrf(){
  const r = await fetch('api.php?action=csrf');
  const j = await r.json();
  CSRF = j.csrf;
}
async function api(path,opts={}){
  const o = Object.assign({headers:{}},opts);
  if (!o.headers['Content-Type'] && o.body) o.headers['Content-Type']='application/json';
  if (!o.headers['X-CSRF-Token']) o.headers['X-CSRF-Token']=CSRF;
  const r = await fetch('api.php'+path,o);
  return r.json();
}
function escapeHtml(s){return (s||'').replace(/[&<>"']/g,function(m){return {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#039;'}[m];});}
function statusLabel(id){
  if (id==1) return 'Backlog';
  if (id==2) return 'Planned';
  if (id==3) return 'In Progress';
  if (id==4) return 'Released';
  if (id==5) return 'Rejected';
  return '';
}
function statusBadge(id){
  const label=statusLabel(id);
  if(!label) return '';
  return '<span class="badge status-badge-'+id+'">'+label+'</span>';
}
let CATS = [];
let FEATS = [];

document.addEventListener('DOMContentLoaded', async ()=>{
  await getCsrf();

  const loginCard = document.getElementById('login-card');
  const adminUI   = document.getElementById('admin-ui');

  async function refreshAdminFlag(){
    const me = await api('?action=admin_me',{headers:{}});
    if (me.user && me.user.role==='admin'){
      loginCard.style.display='none';
      adminUI.style.display='';
      loadSettings();
      loadCats();
      loadFeats();
      loadUsers();
    }
  }
  await refreshAdminFlag();

  document.getElementById('btn-adm-login').addEventListener('click', async ()=>{
    const id = document.getElementById('adm-ident').value.trim();
    const pw = document.getElementById('adm-pass').value;
    const msg = document.getElementById('adm-msg');
    const j = await api('?action=admin_login',{method:'POST',body:JSON.stringify({identifier:id,password:pw})});
    if (j.ok){
      msg.textContent='Logged in.';
      await refreshAdminFlag();
    } else {
      msg.textContent = j.error || 'Login failed.';
    }
  });

  document.getElementById('btn-logout').addEventListener('click', async ()=>{
    await api('?action=logout',{method:'POST',body:'{}'});
    window.location.href='admin.php';
  });

  document.querySelectorAll('.tabbtn').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      document.querySelectorAll('.tabbtn').forEach(b=>b.classList.remove('active'));
      document.querySelectorAll('.tab').forEach(t=>t.classList.remove('active'));
      btn.classList.add('active');
      document.getElementById('tab-'+btn.dataset.tab).classList.add('active');
    });
  });

  async function loadSettings(){
    const j = await api('?action=settings_get',{headers:{}});
    document.getElementById('set-domain').value = j.allowed_domain || '';
  }
  document.getElementById('btn-save-domain').addEventListener('click', async ()=>{
    const val = document.getElementById('set-domain').value.trim();
    const msg = document.getElementById('set-domain-msg');
    const j = await api('?action=settings_set_domain',{method:'POST',body:JSON.stringify({allowed_domain:val})});
    msg.textContent = j.ok ? 'Saved' : (j.error || 'Error');
  });

  async function loadCats(){
    const cats = await api('?action=categories_list',{headers:{}});
    CATS = cats;
    const tbody = document.querySelector('#cat-table tbody');
    tbody.innerHTML = cats.map(c=>'<tr><td>'+escapeHtml(c.name)+'</td><td>'+c.sort_order+'</td><td><button class="btn btn-del-cat" data-id="'+c.id+'">Delete</button></td></tr>').join('');
    const filt = document.getElementById('filt-cat');
    filt.innerHTML = '<option value="">All categories</option>' + cats.map(c=>'<option value="'+c.id+'">'+escapeHtml(c.name)+'</option>').join('');
    document.querySelectorAll('.btn-del-cat').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        if (!confirm('Delete this category?')) return;
        await api('?action=categories_delete',{method:'POST',body:JSON.stringify({id})});
        loadCats();
      });
    });
  }
  document.getElementById('btn-add-cat').addEventListener('click', async ()=>{
    const name = document.getElementById('cat-name').value.trim();
    const sort = Number(document.getElementById('cat-sort').value || '100');
    if (!name) return;
    await api('?action=categories_create',{method:'POST',body:JSON.stringify({name,sort_order:sort})});
    document.getElementById('cat-name').value='';
    loadCats();
  });

  function renderFeatEditRow(id){
    const tbody = document.querySelector('#feat-table tbody');
    tbody.querySelectorAll('.feat-edit-row').forEach(r=>r.remove());
    const feat = FEATS.find(f=>f.id==id);
    if (!feat) return;
    const rows = Array.from(tbody.querySelectorAll('tr'));
    const anchor = rows.find(tr=>{
      const btn = tr.querySelector('.btn-edit-feat');
      return btn && Number(btn.dataset.id)===id;
    });
    if (!anchor) return;

    const tr = document.createElement('tr');
    tr.className = 'feat-edit-row';
    tr.dataset.id = id;
    const td = document.createElement('td');
    td.colSpan = 8;

    const catOptions = ['<option value="">(none)</option>'].concat(
      CATS.map(c=>'<option value="'+c.id+'" '+(c.id==feat.category_id?'selected':'')+'>'+escapeHtml(c.name)+'</option>')
    ).join('');
    const statusOptions = [1,2,3,4,5].map(s=>'<option value="'+s+'" '+(s==feat.status_id?'selected':'')+'>'+statusLabel(s)+'</option>').join('');

    td.innerHTML = `
      <div class="edit-card">
        <div class="edit-row-grid">
          <div>
            <div class="mb8"><strong>${escapeHtml(feat.title)}</strong></div>
            <div class="mb8">
              <div class="muted">Description</div>
              <textarea class="edit-desc" rows="3" readonly>${escapeHtml(feat.description||'')}</textarea>
            </div>
            <div class="muted" style="font-size:12px;">
              Author: ${escapeHtml(feat.author_email||'')}<br>
              Created: ${feat.created_at} · Updated: ${feat.updated_at}<br>
              Votes: ${feat.total_votes||0} ${feat.duplicate_of ? '· Duplicate of #'+feat.duplicate_of : ''}
            </div>
          </div>
          <div>
            <div class="mb8">
              <div class="muted">Status</div>
              <select class="edit-status">${statusOptions}</select>
            </div>
            <div class="mb8">
              <div class="muted">Category</div>
              <select class="edit-cat">${catOptions}</select>
            </div>
            <div class="mb8">
              <button class="btn btn-save-edit">Save changes</button>
              <button class="btn btn-cancel-edit" type="button">Close</button>
            </div>
          </div>
        </div>
      </div>`;
    tr.appendChild(td);
    anchor.after(tr);

    tr.querySelector('.btn-cancel-edit').addEventListener('click', ()=>tr.remove());
    tr.querySelector('.btn-save-edit').addEventListener('click', async ()=>{
      const statusId = Number(tr.querySelector('.edit-status').value);
      const catVal = tr.querySelector('.edit-cat').value;
      const payload = {id:id,status_id:statusId};
      if (catVal!=="") payload.category_id = Number(catVal); else payload.category_id = null;
      await api('?action=admin_feature_update',{method:'POST',body:JSON.stringify(payload)});
      await loadFeats();
    });
  }

  async function loadFeats(){
    const status = document.getElementById('filt-status').value;
    const cat    = document.getElementById('filt-cat').value;
    const q      = document.getElementById('filt-q').value.trim();
    const params = [];
    if (status) params.push('status='+encodeURIComponent(status));
    if (cat) params.push('category_id='+encodeURIComponent(cat));
    if (q) params.push('q='+encodeURIComponent(q));
    const qs = params.length ? ('&'+params.join('&')) : '';
    const data = await api('?action=admin_features_list'+qs,{headers:{}});
    FEATS = data;
    const tbody = document.querySelector('#feat-table tbody');
    tbody.innerHTML = data.map(f=>{
      return '<tr>'+
        '<td>'+f.id+'</td>'+
        '<td>'+escapeHtml(f.title)+'</td>'+
        '<td>'+statusBadge(f.status_id)+'</td>'+
        '<td>'+(f.category_id||'')+'</td>'+
        '<td>'+(f.total_votes||0)+'</td>'+
        '<td>'+escapeHtml(f.author_email||'')+'</td>'+
        '<td>'+(f.duplicate_of||'')+'</td>'+
        '<td>'+
          '<button class="btn btn-edit-feat" data-id="'+f.id+'">Edit</button> '+
          '<button class="btn btn-dup-feat" data-id="'+f.id+'">Mark dup</button> '+
          '<button class="btn btn-del-feat" data-id="'+f.id+'">Del</button>'+
        '</td>'+
      '</tr>';
    }).join('');

    document.querySelectorAll('.btn-del-feat').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        if (!confirm('Delete feature #'+id+'?')) return;
        await api('?action=admin_feature_delete',{method:'POST',body:JSON.stringify({id})});
        loadFeats();
      });
    });
    document.querySelectorAll('.btn-dup-feat').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        const dup = prompt('Mark feature #'+id+' as duplicate of feature id:');
        if (!dup) return;
        await api('?action=features_mark_duplicate',{method:'POST',body:JSON.stringify({feature_id:id,duplicate_of_id:Number(dup)})});
        loadFeats();
      });
    });
    document.querySelectorAll('.btn-edit-feat').forEach(b=>{
      b.addEventListener('click', ()=>{
        const id = Number(b.dataset.id);
        renderFeatEditRow(id);
      });
    });
  }
  document.getElementById('btn-filt').addEventListener('click', loadFeats);

  async function loadUsers(){
    const data = await api('?action=admin_users_list',{headers:{}});
    const tbody = document.querySelector('#user-table tbody');
    tbody.innerHTML = data.map(u=>{
      const tick = u.role==='admin' ? '✔' : '';
      const ls = u.last_seen_at ? new Date(u.last_seen_at.replace(' ','T')+'Z').toLocaleDateString() : '';
      return '<tr>'+
        '<td>'+escapeHtml(u.email)+'</td>'+
        '<td style="text-align:center">'+tick+'</td>'+
        '<td>'+ls+'</td>'+
        '<td>'+(u.cnt_backlog||0)+'</td>'+
        '<td>'+(u.cnt_planned||0)+'</td>'+
        '<td>'+(u.cnt_progress||0)+'</td>'+
        '<td>'+(u.cnt_released||0)+'</td>'+
        '<td>'+(u.cnt_rejected||0)+'</td>'+
        '<td><button class="btn btn-toggle-admin" data-email="'+escapeHtml(u.email)+'">Toggle admin</button> '+
            '<button class="btn btn-view-user" data-id="'+u.id+'">View</button></td>'+
      '</tr>';
    }).join('');
    document.querySelectorAll('.btn-toggle-admin').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const email = b.dataset.email;
        const row = data.find(u=>u.email===email);
        const newRole = row.role==='admin' ? 'user' : 'admin';
        await api('?action=users_set_role',{method:'POST',body:JSON.stringify({email,role:newRole})});
        loadUsers();
      });
    });
    document.querySelectorAll('.btn-view-user').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        const info = await api('?action=admin_user_detail&id='+id,{headers:{}});
        const ud = document.getElementById('user-detail');
        const u = info.user;
        const feats = info.features || [];
        let html = '<h3>User detail</h3>';
        html += '<p><strong>'+escapeHtml(u.email)+'</strong> ('+escapeHtml(u.name||'')+') – '+(u.role==='admin'?'Admin':'Standard')+'</p>';
        html += '<p class="muted">Last seen: '+(u.last_seen_at||'n/a')+'</p>';
        if (!feats.length){
          html += '<p class="muted">No submissions yet.</p>';
        } else {
          html += '<table><thead><tr><th>ID</th><th>Title</th><th>Status</th><th>Created</th><th>Updated</th></tr></thead><tbody>';
          html += feats.map(f=>'<tr><td>'+f.id+'</td><td>'+escapeHtml(f.title)+'</td><td>'+statusLabel(f.status_id)+'</td><td>'+f.created_at+'</td><td>'+f.updated_at+'</td></tr>').join('');
          html += '</tbody></table>';
        }
        ud.innerHTML = html;
      });
    });
  }

  document.getElementById('btn-create-admin').addEventListener('click', async ()=>{
    const email = document.getElementById('u-email').value.trim();
    const name  = document.getElementById('u-name').value.trim();
    const pass  = document.getElementById('u-pass').value;
    if (!email || !pass) return;
    await api('?action=users_create_admin',{method:'POST',body:JSON.stringify({email,name,password:pass})});
    document.getElementById('u-email').value='';
    document.getElementById('u-name').value='';
    document.getElementById('u-pass').value='';
    loadUsers();
  });

});
</script>
</body>
</html>
