<?php
require __DIR__ . '/../db.php';
$u = current_user($db);
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>Feature Requests</title>
<style>
/* Brand + palette */
:root {
  /* Your palette */
  --color-1: #ffffff;
  --color-2: #afbac5;
  --color-3: #7f9aad;
  --color-4: #5c92c1;
  --color-5: #7890a1;

  /* Derived app colors */
  --primary: var(--color-4);
  --primary-dark: #325a8b;      /* deeper AGR blue */
  --bg-soft: #f3f5f8;
  --card-bg: var(--color-1);
  --border-soft: #dde3ea;
  --text-main: #2f343b;
  --text-muted: #6c7175;
}

body{
  font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu;
  margin:0;
  background:var(--bg-soft);
  color:var(--text-main);
}
.wrap{
  max-width:960px;
  margin:24px auto;
  padding:0 16px;
}
.card{
  background:var(--card-bg);
  border:1px solid var(--border-soft);
  border-radius:12px;
  box-shadow:0 2px 6px rgba(0,0,0,.04);
  padding:16px;
}
.row{
  display:flex;
  gap:12px;
  align-items:center;
  flex-wrap:wrap;
}

/* Primary button */
.btn{
  border:1px solid var(--primary);
  background:var(--primary);
  color:#fff;
  border-radius:10px;
  padding:8px 12px;
  cursor:pointer;
  transition:background .15s,border-color .15s,transform .05s;
}
.btn:hover{
  background:var(--primary-dark);
  border-color:var(--primary-dark);
  transform:translateY(-1px);
}

/* Optional neutral button */
.btn-secondary{
  border:1px solid var(--border-soft);
  background:var(--card-bg);
  color:var(--text-main);
}
.btn-secondary:hover{
  background:#f3f5f8;
}

.vote{
  min-width:50px;
  text-align:center;
}
.muted{
  color:var(--text-muted);
}
.mb8{margin-bottom:8px}
.mb16{margin-bottom:16px}

input,textarea,select{
  width:100%;
  border:1px solid var(--border-soft);
  border-radius:10px;
  padding:8px;
}

ul{
  list-style:none;
  padding:0;
  margin:0;
}
li{margin-bottom:12px}

.badge{
  font-size:12px;
  border:1px solid var(--border-soft);
  border-radius:999px;
  padding:2px 8px;
  margin-left:6px;
}

/* Tabs for submit / my submissions */
.tabs{
  display:flex;
  gap:8px;
  margin-bottom:12px;
}
.tabbtn{
  padding:6px 10px;
  border-radius:999px;
  border:1px solid var(--border-soft);
  background:#f9fafb;
  cursor:pointer;
  font-size:14px;
  color:var(--text-main);
}
.tabbtn.active{
  background:var(--primary);
  color:#f9fafb;
  border-color:var(--primary);
}
.tab{display:none}
.tab.active{display:block}

/* My submissions table */
table{
  border-collapse:collapse;
  width:100%;
  font-size:14px;
}
th,td{
  border-bottom:1px solid var(--border-soft);
  padding:6px 4px;
  text-align:left;
}
th{font-weight:600}
.site-header{
  background:var(--card-bg);
  border-bottom:1px solid var(--border-soft);
  box-shadow:0 2px 4px rgba(0,0,0,.03);
}
.header-inner{
  max-width:960px;
  margin:0 auto;
  padding:10px 16px;
  display:flex;
  align-items:center;
  gap:16px;
}
.header-inner img{
  height:40px;
}
.header-title{
  font-size:20px;
  font-weight:600;
  color:var(--primary);
}

/* Sort buttons: neutral by default, colored when active */
.btn.sort {
  background:#f9fafb;
  color:var(--text-main);
  border-color:var(--border-soft);
}

.btn.sort.active {
  background:var(--primary);
  color:#fff;
  border-color:var(--primary);
  transform:translateY(-1px);
}

/* Exact duplicate suggestion styling */
.dup-item.exact {
  font-weight:600;
  color:var(--primary-dark);
}

.badge-exact {
  display:inline-block;
  margin-left:4px;
  padding:2px 6px;
  border-radius:999px;
  font-size:11px;
  background:var(--primary);
  color:#fff;
}


</style>

</head>
<body>
<header class="site-header">
  <div class="header-inner">
    <img src="../images/ailogo.png"
         alt="AGRIntelligence logo"
         onerror="this.style.display='none'">
  </div>
</header>

<div class="wrap">
  <h1 class="mb16">Feature Requests</h1>

  <?php if (!$u): ?>
    <div class="card">
      <h2>Sign in</h2>
      <p class="muted">Enter your work email and we&rsquo;ll email you a magic link.</p>
      <div class="row mb8">
        <input id="ml-email" placeholder="your.name@helenaagri.com" style="max-width:320px" />
        <button class="btn" id="btn-ml">Send magic link</button>
      </div>
      <div id="ml-msg" class="muted"></div>
      <p class="muted" style="margin-top:12px;font-size:13px;">Access is limited to approved domains. Links expire in about 15 minutes.</p>
    </div>
  <?php else: ?>
   <div class="card mb16">
      <div class="row mb8" style="justify-content:space-between;align-items:center;">
        <div class="row" style="gap:8px;align-items:center;flex-wrap:nowrap;">
          <div class="muted">
            Signed in as <strong><?php echo htmlspecialchars($u['email']); ?></strong>
          </div>
          <button class="btn" id="btn-logout">Log out</button>
        </div>
        <div id="submission-usage" class="muted" style="font-size:13px;"></div>
      </div>
    
      <div class="tabs">
        <button class="tabbtn active" data-tab="submit">Submit idea</button>
        <button class="tabbtn" data-tab="mine">My submissions</button>
      </div>

      <div class="tab active" id="tab-submit">
        <form id="idea-form">
          <div class="mb8"><input id="idea-title" placeholder="Short title (e.g., Bulk CSV import)"></div>
          <div class="mb8"><textarea id="idea-desc" rows="3" placeholder="Optional details"></textarea></div>
          <div class="mb8"><select id="idea-category"></select></div>
          <div class="mb8">
            <small class="muted">We&rsquo;ll check for duplicates as you type.</small>
            <ul id="dups" class="muted"></ul>
          </div>
          <button class="btn" id="btn-submit-idea" type="submit">Submit idea</button>
          <span class="muted" id="status-msg"></span>
        </form>
      </div>

      <div class="tab" id="tab-mine">
        <div id="mine-empty" class="muted">No previous feature submissions yet.</div>
        <table id="mine-table" style="display:none;margin-top:8px;">
          <thead>
            <tr>
              <th>ID</th><th>Title</th><th>Status</th><th>Votes</th><th>Created</th>
            </tr>
          </thead>
          <tbody></tbody>
        </table>
      </div>
    </div>

    <div class="row mb16">
      <button class="btn sort" data-sort="top">Top</button>
      <button class="btn sort" data-sort="trending">Trending</button>
      <button class="btn sort" data-sort="new">New</button>
      <select id="filter-category" class="btn"></select>
    </div>

    <ul id="ideas"></ul>

    <div class="card mb16">
      <a class="btn" href="roadmap.php">Open Roadmap</a>
    </div>

    <!-- Duplicate feature modal -->
    <div id="dup-modal"
         style="display:none;position:fixed;inset:0;background:rgba(0,0,0,.4);z-index:1000;">
      <div style="background:#fff;max-width:520px;margin:80px auto;padding:16px;
                  border-radius:12px;box-shadow:0 10px 25px rgba(0,0,0,.25);">
        <h3 id="dup-modal-title" style="margin-top:0;margin-bottom:4px;"></h3>
        <div id="dup-modal-meta" class="muted" style="font-size:13px;margin-bottom:8px;"></div>
        <div id="dup-modal-desc"
             style="white-space:pre-wrap;font-size:14px;margin-bottom:12px;"></div>
        <div class="row">
          <button class="btn" id="dup-modal-vote" type="button">
            Vote for this feature
          </button>
          <button class="btn btn-secondary" id="dup-modal-close" type="button">
            Close
          </button>
        </div>
      </div>
    </div>

  <?php endif; ?>
</div>

<script>
let CSRF = null;
async function getCsrf(){
  const r = await fetch('api.php?action=csrf');
  const j = await r.json();
  CSRF = j.csrf;
}
async function api(path,opts={}) {
  const o = Object.assign({headers:{}}, opts);
  if (!o.headers['Content-Type'] && o.body) {
    o.headers['Content-Type'] = 'application/json';
  }
  if (!o.headers['X-CSRF-Token']) {
    o.headers['X-CSRF-Token'] = CSRF;
  }

  let r = await fetch('api.php' + path, o);
  let j = await r.json();

  if (j && j.error === 'csrf_failed') {
    try {
      await getCsrf();
      o.headers['X-CSRF-Token'] = CSRF;
      r = await fetch('api.php' + path, o);
      j = await r.json();
    } catch (e) {}
    if (j && j.error === 'csrf_failed') {
      alert(j.message || 'Your session has expired. Please refresh the page and try again.');
    }
  }

  return j;
}

function escapeHtml(s){
  return (s||'').replace(/[&<>"']/g,function(m){
    return {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#039;'}[m];
  });
}

// simple debounce helper for duplicate checking
function debounce(fn, delay) {
  let t;
  return function(...args) {
    clearTimeout(t);
    t = setTimeout(() => fn.apply(this, args), delay);
  };
}

<?php if (!$u): ?>
// =========================
// Logged-out: magic link UI
// =========================
document.addEventListener('DOMContentLoaded', async () => {
  await getCsrf();
  document.getElementById('btn-ml').addEventListener('click', async () => {
    const email = document.getElementById('ml-email').value.trim();
    const msg = document.getElementById('ml-msg');
    if (!email){
      msg.textContent='Enter your email.';
      return;
    }
    const j = await api('?action=request_magic_link',{
      method:'POST',
      body:JSON.stringify({email})
    });
    msg.textContent = j.ok
      ? 'Check your email for a sign-in link.'
      : (j.error || 'Unable to send link.');
  });
});
<?php else: ?>
// =========================
// Logged-in UI
// =========================
let currentSort = 'top';

function statusLabel(id){
  if (id==1) return 'Backlog';
  if (id==2) return 'Planned';
  if (id==3) return 'In Progress';
  if (id==4) return 'Released';
  if (id==5) return 'Rejected';
  return '';
}

// modal globals
let dupModal, dupModalTitle, dupModalMeta, dupModalDesc, dupModalVote, dupModalClose;
let dupModalFeatureId = null;

async function openDupModal(id) {
  dupModalFeatureId = id;
  const f = await api('?action=feature_get&id=' + id, { headers: {} });

  if (!f || f.error) {
    alert(f && f.error ? f.error : 'Could not load feature details.');
    return;
  }

  dupModalTitle.textContent = f.title || '(untitled)';
  const parts = [];
  if (f.category_name) parts.push(f.category_name);
  if (typeof f.total_votes !== 'undefined') parts.push((f.total_votes || 0) + ' votes');
  if (typeof f.status_id !== 'undefined') parts.push('Status: ' + statusLabel(f.status_id));
  dupModalMeta.textContent = parts.join(' · ');
  dupModalDesc.textContent = f.description || '(no description provided)';

  dupModal.style.display = 'block';
}

async function loadCategories(){
  const cats = await api('?action=categories_list',{headers:{}});
  const sel = document.getElementById('idea-category');
  const filt = document.getElementById('filter-category');
  sel.innerHTML = '<option value="">Select category (optional)</option>' +
    cats.map(c=>`<option value="${c.id}">${escapeHtml(c.name)}</option>`).join('');
  filt.innerHTML = '<option value="">All categories</option>' +
    cats.map(c=>`<option value="${c.id}">${escapeHtml(c.name)}</option>`).join('');
}


async function loadSubmissionUsage() {
  const el = document.getElementById('submission-usage');
  const submitBtn = document.getElementById('btn-submit-idea');
  const statusMsg = document.getElementById('status-msg');

  if (!el) return;

  const j = await api('?action=my_submission_usage', { headers: {} });

  if (!j || !j.ok) {
    el.textContent = '';
    if (submitBtn) submitBtn.disabled = false;
    return;
  }

  const limit = j.limit ?? 0;
  const used  = j.used ?? 0;

  if (!limit || limit <= 0) {
    // Unlimited
    el.textContent = 'You can submit unlimited feature ideas.';
    if (submitBtn) {
      submitBtn.disabled = false;
      submitBtn.textContent = 'Submit idea';
    }
    if (statusMsg) statusMsg.textContent = '';
    return;
  }

  const remaining = Math.max(0, limit - used);
  el.textContent = `You have used ${used} of ${limit} active submissions (${remaining} remaining).`;

  if (submitBtn) {
    if (remaining <= 0) {
      submitBtn.disabled = true;
      submitBtn.textContent = 'Limit reached';
      if (statusMsg) {
        statusMsg.textContent = 'You have reached the maximum number of active submissions.';
      }
    } else {
      submitBtn.disabled = false;
      submitBtn.textContent = 'Submit idea';
      if (statusMsg && statusMsg.textContent === 'You have reached the maximum number of active submissions.') {
        statusMsg.textContent = '';
      }
    }
  }
}




async function loadIdeas(){
  const catId = document.getElementById('filter-category').value;
  const qs = '?action=features_list'
           + '&sort=' + encodeURIComponent(currentSort)
           + (catId ? ('&category_id='+encodeURIComponent(catId)) : '');
  const data = await api(qs,{headers:{}});

  const ul = document.getElementById('ideas');
  ul.innerHTML = data.map(it=>{
    const votes = it.total_votes ?? 0;
    const badge = statusLabel(it.status_id)
      ? `<span class="badge">${statusLabel(it.status_id)}</span>` : '';
    const cat = it.category_name
      ? `<span class="badge">${escapeHtml(it.category_name)}</span>` : '';
    const dup = it.duplicate_of
      ? `<span class="badge">Dup of #${it.duplicate_of}</span>` : '';
    const dt = new Date(it.created_at.replace(' ','T')+'Z').toLocaleString();
    return `<li class="card">
      <div class="row">
        <div class="vote">
          <button class="btn vote-btn" data-id="${it.id}">▲</button>
          <div>${votes}</div>
        </div>
        <div>
          <div><strong>${escapeHtml(it.title)}</strong> ${badge} ${cat} ${dup}</div>
          <div class="muted">${escapeHtml(it.description||'')}</div>
          <div class="muted" style="font-size:12px">${dt}</div>
        </div>
      </div>
    </li>`;
  }).join('');

  // hook up voting
  document.querySelectorAll('.vote-btn').forEach(b=>{
    b.addEventListener('click', async ()=>{
      const j = await api('?action=vote_toggle',{
        method:'POST',
        body:JSON.stringify({feature_id:Number(b.dataset.id)})
      });
      if (j.ok) {
        loadIdeas();
      } else {
        alert(j.error||'Vote failed');
      }
    });
  });
}

async function loadMine(){
  const data = await api('?action=my_features',{method:'POST',body:'{}'});
  const empty = document.getElementById('mine-empty');
  const table = document.getElementById('mine-table');
  const tbody = table.querySelector('tbody');
  if (!data.length){
    empty.style.display='block';
    table.style.display='none';
    return;
  }
  empty.style.display='none';
  table.style.display='table';
  tbody.innerHTML = data.map(it=>{
    const dt = new Date(it.created_at.replace(' ','T')+'Z').toLocaleDateString();
    return `<tr>
      <td>${it.id}</td>
      <td>${escapeHtml(it.title)}</td>
      <td>${statusLabel(it.status_id)}</td>
      <td>${it.total_votes ?? 0}</td>
      <td>${dt}</td>
    </tr>`;
  }).join('');
}

function renderDupSuggestions(list, container) {
  if (!list || !list.length) {
    container.innerHTML = '';
    return;
  }

  container.innerHTML = list.map(d => {
    const isExact = (d.distance === 0);
    return `
      <li class="dup-item${isExact ? ' exact' : ''}"
          data-id="${d.id}"
          style="cursor:pointer;">
        Close match: “${escapeHtml(d.title)}”
        ${
          isExact
            ? '<span class="badge-exact">Exact match</span>'
            : `<span class="muted">(distance ${d.distance})</span>`
        }
      </li>
    `;
  }).join('');
}


document.addEventListener('DOMContentLoaded', async ()=>{
  await getCsrf();
  await loadCategories();
  await loadIdeas();
  await loadMine();
  await loadSubmissionUsage();

  // wire modal elements
  dupModal       = document.getElementById('dup-modal');
  dupModalTitle  = document.getElementById('dup-modal-title');
  dupModalMeta   = document.getElementById('dup-modal-meta');
  dupModalDesc   = document.getElementById('dup-modal-desc');
  dupModalVote   = document.getElementById('dup-modal-vote');
  dupModalClose  = document.getElementById('dup-modal-close');

  if (dupModal && dupModalClose) {
    dupModalClose.addEventListener('click', () => {
      dupModal.style.display = 'none';
    });
    // click backdrop to close
    dupModal.addEventListener('click', (e) => {
      if (e.target === dupModal) {
        dupModal.style.display = 'none';
      }
    });
  }

  if (dupModalVote) {
    dupModalVote.addEventListener('click', async () => {
      if (!dupModalFeatureId) return;
      const j = await api('?action=vote_toggle', {
        method: 'POST',
        body: JSON.stringify({ feature_id: dupModalFeatureId })
      });
      if (j && j.ok) {
        dupModal.style.display = 'none';
        loadIdeas();
        loadMine();
      } else {
        alert((j && j.error) || 'Vote failed');
      }
    });
  }

  // sort buttons: Top / Trending / New
  document.querySelectorAll('.sort').forEach(b=>{
    b.addEventListener('click', ()=>{
      document.querySelectorAll('.sort').forEach(x => x.classList.remove('active'));
      b.classList.add('active');
      currentSort = b.dataset.sort || 'top';
      loadIdeas();
    });
  });

  // category filter
  document.getElementById('filter-category').addEventListener('change', loadIdeas);

  // logout
  document.getElementById('btn-logout').addEventListener('click', async ()=>{
    await api('?action=logout',{method:'POST',body:'{}'});
    window.location.href = 'index.php';
  });

  // duplicate checking on title input (debounced)
  const titleEl = document.getElementById('idea-title');
  const dupList = document.getElementById('dups');

  if (titleEl && dupList) {
    const checkDup = debounce(async () => {
      const q = titleEl.value.trim();
      if (q.length < 3) {
        dupList.innerHTML = '';
        return;
      }

      const data = await api(
        '?action=duplicate_suggest_lev&q=' + encodeURIComponent(q),
        { headers: {} }
      );

      if (!Array.isArray(data) || data.length === 0) {
        dupList.innerHTML = '';
        return;
      }

      renderDupSuggestions(data, dupList);

      // make each suggestion clickable -> opens modal
      dupList.querySelectorAll('.dup-item').forEach(li => {
        li.addEventListener('click', () => {
          const id = Number(li.dataset.id);
          if (id) openDupModal(id);
        });
      });
    }, 300);

    titleEl.addEventListener('input', checkDup);
  }

  // submit idea
  document.getElementById('idea-form').addEventListener('submit', async e=>{
    e.preventDefault();
    const title = document.getElementById('idea-title').value.trim();
    const desc  = document.getElementById('idea-desc').value.trim();
    const catId = document.getElementById('idea-category').value || null;
    const msg   = document.getElementById('status-msg');
    if (!title){
      msg.textContent='Title required';
      return;
    }
    const j = await api('?action=features_create',{
      method:'POST',
      body:JSON.stringify({
        title,
        description:desc,
        category_id: catId ? Number(catId) : null
      })
    });
    if (j.ok){
      e.target.reset();
      dupList.innerHTML='';
      msg.textContent='Submitted!';
      loadIdeas();
      loadMine();
      loadSubmissionUsage();
    } else {
      msg.textContent = j.message || j.error || 'Failed';
    }
  });

  // tab switching: Submit / My submissions
  document.querySelectorAll('.tabbtn').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      document.querySelectorAll('.tabbtn').forEach(b=>b.classList.remove('active'));
      document.querySelectorAll('.tab').forEach(t=>t.classList.remove('active'));
      btn.classList.add('active');
      document.getElementById('tab-'+btn.dataset.tab).classList.add('active');
    });
  });
});
<?php endif; ?>
</script>

</body>
</html>
